import SwiftUI

/// View for managing data retention settings
struct DataRetentionView: View {
    @StateObject private var privacyService = PrivacyService.shared
    @State private var selectedRetention: PrivacyService.RetentionPeriod
    @Environment(\.dismiss) private var dismiss
    
    init() {
        _selectedRetention = State(initialValue: PrivacyService.shared.currentRetentionPeriod)
    }
    
    var body: some View {
        NavigationView {
            VStack(alignment: .leading, spacing: 20) {
                // Header Information
                VStack(alignment: .leading, spacing: 8) {
                    Text("Data Retention")
                        .font(.title2)
                        .fontWeight(.bold)
                    
                    Text("Choose how long to keep your personal data. Older data will be automatically deleted according to your preference.")
                        .font(.caption)
                        .foregroundColor(.secondary)
                }
                .padding(.horizontal)
                
                // Retention Period Selection
                VStack(alignment: .leading, spacing: 12) {
                    Text("Retention Period")
                        .font(.headline)
                        .padding(.horizontal)
                    
                    ForEach(PrivacyService.RetentionPeriod.allCases, id: \.self) { period in
                        retentionOptionView(for: period)
                    }
                }
                
                Spacer()
                
                // Information Section
                VStack(alignment: .leading, spacing: 8) {
                    Text("Important Information")
                        .font(.headline)
                        .padding(.horizontal)
                    
                    VStack(alignment: .leading, spacing: 4) {
                        InfoRow(icon: "shield.checkered", text: "Data is stored securely on your device")
                        InfoRow(icon: "clock", text: "Automatic cleanup happens in the background")
                        InfoRow(icon: "arrow.counterclockwise", text: "You can change this setting anytime")
                        InfoRow(icon: "exclamationmark.triangle", text: "Deleted data cannot be recovered")
                    }
                    .padding(.horizontal)
                }
                
                // Apply Changes Button
                Button(action: applyRetentionSetting) {
                    HStack {
                        Image(systemName: "checkmark.circle")
                        Text("Apply Setting")
                    }
                    .frame(maxWidth: .infinity)
                    .padding()
                    .background(Color.accentColor)
                    .foregroundColor(.white)
                    .cornerRadius(10)
                }
                .padding(.horizontal)
                .disabled(selectedRetention == privacyService.currentRetentionPeriod)
            }
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .navigationBarLeading) {
                    Button("Cancel") { dismiss() }
                }
            }
        }
    }
    
    @ViewBuilder
    private func retentionOptionView(for period: PrivacyService.RetentionPeriod) -> some View {
        Button(action: { selectedRetention = period }) {
            HStack {
                VStack(alignment: .leading, spacing: 2) {
                    Text(period.displayName)
                        .font(.body)
                        .fontWeight(.medium)
                    
                    if let days = period.days {
                        Text("Data older than \(days) days will be deleted")
                            .font(.caption)
                            .foregroundColor(.secondary)
                    } else {
                        Text("Data will be kept until manually deleted")
                            .font(.caption)
                            .foregroundColor(.secondary)
                    }
                }
                
                Spacer()
                
                Image(systemName: selectedRetention == period ? "checkmark.circle.fill" : "circle")
                    .foregroundColor(selectedRetention == period ? .accentColor : .secondary)
            }
            .padding()
            .background(
                RoundedRectangle(cornerRadius: 10)
                    .fill(selectedRetention == period ? Color.accentColor.opacity(0.1) : Color.clear)
            )
            .overlay(
                RoundedRectangle(cornerRadius: 10)
                    .stroke(selectedRetention == period ? Color.accentColor : Color.gray.opacity(0.3), lineWidth: 1)
            )
        }
        .buttonStyle(PlainButtonStyle())
        .padding(.horizontal)
    }
    
    private func applyRetentionSetting() {
        privacyService.currentRetentionPeriod = selectedRetention
        
        // Provide haptic feedback
        WKInterfaceDevice.current().play(.success)
        
        dismiss()
    }
}

struct InfoRow: View {
    let icon: String
    let text: String
    
    var body: some View {
        HStack(spacing: 8) {
            Image(systemName: icon)
                .foregroundColor(.accentColor)
                .frame(width: 16)
            
            Text(text)
                .font(.caption)
                .foregroundColor(.secondary)
        }
    }
}

#Preview {
    DataRetentionView()
}